// ===== Global Variables =====
let currentSlide = 0;
let slideInterval;
const totalSlides = 3;

// ===== Hero Slider Function =====
function initSlider() {
    const slides = document.querySelectorAll('.slide');
    const sliderDots = document.querySelector('.slider-dots');

    if (slides.length === 0) return;

    // Create dots
    sliderDots.innerHTML = '';
    slides.forEach((slide, index) => {
        const dot = document.createElement('span');
        dot.classList.add('dot');
        if (index === 0) dot.classList.add('active');
        dot.onclick = () => goToSlide(index);
        sliderDots.appendChild(dot);
    });

    // Show first slide
    showSlide(0);

    // Start auto-play
    startAutoPlay();
}

function showSlide(n) {
    const slides = document.querySelectorAll('.slide');
    const dots = document.querySelectorAll('.dot');

    if (slides.length === 0) return;

    // Loop back to start or end
    if (n >= slides.length) {
        currentSlide = 0;
    } else if (n < 0) {
        currentSlide = slides.length - 1;
    } else {
        currentSlide = n;
    }

    // Hide all slides and remove active class
    slides.forEach((slide, index) => {
        slide.style.display = 'none';
        slide.classList.remove('active');
    });

    // Remove active from all dots
    dots.forEach(dot => {
        dot.classList.remove('active');
    });

    // Show current slide with active class
    if (slides[currentSlide]) {
        slides[currentSlide].style.display = 'block';
        slides[currentSlide].classList.add('active');
    }

    // Activate current dot
    if (dots[currentSlide]) {
        dots[currentSlide].classList.add('active');
    }
}

function changeSlide(direction) {
    stopAutoPlay();
    const newSlide = currentSlide + direction;
    showSlide(newSlide);
    startAutoPlay();
}

function goToSlide(n) {
    stopAutoPlay();
    showSlide(n);
    startAutoPlay();
}

function nextSlide() {
    showSlide(currentSlide + 1);
}

function prevSlide() {
    showSlide(currentSlide - 1);
}

function startAutoPlay() {
    stopAutoPlay(); // Clear any existing interval
    slideInterval = setInterval(() => {
        nextSlide();
    }, 5000);
}

function stopAutoPlay() {
    if (slideInterval) {
        clearInterval(slideInterval);
    }
}

// ===== Hamburger Menu =====
function initHamburger() {
    const hamburger = document.querySelector('.hamburger');
    const navMenu = document.querySelector('.nav-menu');

    if (!hamburger || !navMenu) return;

    hamburger.addEventListener('click', () => {
        hamburger.classList.toggle('active');
        navMenu.classList.toggle('active');
    });

    // Close menu when clicking links
    document.querySelectorAll('.nav-menu a').forEach(link => {
        link.addEventListener('click', () => {
            hamburger.classList.remove('active');
            navMenu.classList.remove('active');
        });
    });

    // Close menu when clicking outside
    document.addEventListener('click', (e) => {
        if (!hamburger.contains(e.target) && !navMenu.contains(e.target)) {
            hamburger.classList.remove('active');
            navMenu.classList.remove('active');
        }
    });
}

// ===== Gallery Filter =====
function initGalleryFilter() {
    const filterBtns = document.querySelectorAll('.filter-btn');
    const galleryItems = document.querySelectorAll('.gallery-item');

    if (filterBtns.length === 0) return;

    filterBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            // Remove active from all
            filterBtns.forEach(b => b.classList.remove('active'));
            btn.classList.add('active');

            const filter = btn.getAttribute('data-filter');

            galleryItems.forEach(item => {
                if (filter === 'all' || item.getAttribute('data-category') === filter) {
                    item.style.display = 'block';
                    item.style.animation = 'fadeIn 0.5s ease';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    });
}

// ===== FAQ Accordion =====
function initFAQ() {
    const faqItems = document.querySelectorAll('.faq-item');

    faqItems.forEach(item => {
        const question = item.querySelector('.faq-question');

        if (question) {
            question.addEventListener('click', () => {
                // Close others
                faqItems.forEach(other => {
                    if (other !== item) {
                        other.classList.remove('active');
                    }
                });

                // Toggle current
                item.classList.toggle('active');
            });
        }
    });
}

// ===== Contact Form =====
function initContactForm() {
    const form = document.getElementById('contactForm');

    if (!form) return;

    form.addEventListener('submit', (e) => {
        e.preventDefault();

        const name = document.getElementById('name').value;
        const email = document.getElementById('email').value;

        alert(`Thank you ${name}! We received your message and will contact you at ${email} soon.`);
        form.reset();
    });
}

// ===== Scroll Animations =====
function initScrollAnimations() {
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    });

    const elements = document.querySelectorAll('.service-card, .why-card, .value-card, .team-member, .gallery-item');
    elements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(30px)';
        el.style.transition = 'all 0.6s ease';
        observer.observe(el);
    });
}

// ===== Navbar Scroll Effect =====
function initNavbarScroll() {
    window.addEventListener('scroll', () => {
        const navbar = document.querySelector('.navbar');
        if (navbar) {
            if (window.scrollY > 50) {
                navbar.style.boxShadow = '0 5px 20px rgba(0,0,0,0.15)';
                navbar.style.padding = '10px 0';
            } else {
                navbar.style.boxShadow = '0 2px 10px rgba(0,0,0,0.1)';
                navbar.style.padding = '15px 0';
            }
        }
    });
}

// ===== Smooth Scroll =====
function initSmoothScroll() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

// ===== Add Animation Styles =====
const style = document.createElement('style');
style.textContent = `
    @keyframes fadeIn {
        from {
            opacity: 0;
            transform: scale(0.95);
        }
        to {
            opacity: 1;
            transform: scale(1);
        }
    }

    .slide {
        transition: opacity 0.5s ease-in-out;
    }
`;
document.head.appendChild(style);

// ===== Initialize Everything When Page Loads =====
document.addEventListener('DOMContentLoaded', () => {
    console.log('%c🍽️ Delicious Catering', 'color: #e74c3c; font-size: 24px; font-weight: bold;');
    console.log('%c✅ Initializing website...', 'color: #27ae60; font-size: 14px;');

    // Initialize all functions
    initSlider();
    initHamburger();
    initGalleryFilter();
    initFAQ();
    initContactForm();
    initScrollAnimations();
    initNavbarScroll();
    initSmoothScroll();

    console.log('%c✅ All features loaded successfully!', 'color: #27ae60; font-size: 14px; font-weight: bold;');
});

// ===== Keyboard Navigation for Slider =====
document.addEventListener('keydown', (e) => {
    if (e.key === 'ArrowLeft') {
        changeSlide(-1);
    }
    if (e.key === 'ArrowRight') {
        changeSlide(1);
    }
});

// ===== Stop slider when page is hidden =====
document.addEventListener('visibilitychange', () => {
    if (document.hidden) {
        stopAutoPlay();
    } else {
        startAutoPlay();
    }
});

// ===== Pause slider on hover =====
document.addEventListener('DOMContentLoaded', () => {
    const sliderContainer = document.querySelector('.hero-slider');
    if (sliderContainer) {
        sliderContainer.addEventListener('mouseenter', stopAutoPlay);
        sliderContainer.addEventListener('mouseleave', startAutoPlay);
    }
});
